/*
  Sonometre_Nano_v2.ino
  Version optimisée pour ruban LED avec micro MAX4466 
  Ecrit le 21/07/2025 pour carte Arduino Nano v3
  Améliorations : lissage du signal par calcul de la moyenne exponentielle, 
                  calibration dynamique, 
                  transitions fluides des LEDs.
  Utilisation de la bibliothèque FastLED version 3.9.16
  IMPORTANT : réglage fin du gain par potentiomètre au dos du micro MAX4466.
*/

#define TRACE 0   // mise au point : 1 pour activer trace sur port série, 0 pour désactiver
//--------------------------------------------------------------------------
// Configuration
#define NB_LEDS   37       // Nombre de LEDs sur le ruban (0 à 36)
#define MICRO_PIN A5       // Entrée microphone
#define LED_PIN   2        // Pin de commande des LEDs
#define DURECHANT 50       // Durée d'échantillonnage (ms)
#define ALPHA 0.5          // Facteur de lissage exponentiel (0.0 à 0.5)

//--------------------------------------------------------------------------
#include <FastLED.h>
CRGB led[NB_LEDS];

// Variables globales
float signallisse = 0;    // Valeur lissée du signal
float nivrepos = 512;     // Niveau de repos estimé du microphone (Tension alim/2)
int indmic = 0;           // Indice de la LED correspondant au niveau sonore
int hue = 0;              // Couleur des LEDs

//---------------------------------------------
// Éteint les LEDs au-delà de l'indice
void stopLeds(int startIndex) {
  for (int i = startIndex; i < NB_LEDS; i++) {
    led[i] = CRGB::Black;
  }
  FastLED.show();
}

//---------------------------------------------
// Animation des LEDs au démarrage
void animdepart() {
  for (int i = 0; i < NB_LEDS; i++) {
    int hval = random(0, 255); // Couleur aléatoire
    led[i] = CHSV(hval, 255, 255);
    FastLED.show();
    delay(30);
  }
  for (int i = NB_LEDS - 1; i >= 0; i--) {
    led[i] = CRGB::Black;
    FastLED.show();
    delay(30);
  }
}

//---------------------------------------------
// Calcule l'écart maximal entre 2 pics sur la durée d'échantillonage
int mesurepicapic() {
  unsigned long startMillis = millis();
  int signalmax = 0;
  int signalmin = 1023;
  int echantillon;

  // Échantillonnage du signal pendant durée prévue (max et min des valeurs lues)
  while (millis() - startMillis < DURECHANT) {
    echantillon = analogRead(MICRO_PIN);
    if (echantillon < 1023) {         // Ignore les valeurs aberrantes
      if (echantillon > signalmax) signalmax = echantillon;
      if (echantillon < signalmin) signalmin = echantillon;
    }
  }
  return signalmax - signalmin; // renvoie l'amplitude pic-à-pic
}

//---------------------------------------------
// Mise à jour du niveau moyen
// moyenne exponentielle (nouvelle_moyenne = (ancienne_moyenne × (1 - α)) + (nouvelle_valeur_lue × α))
void majvalrepos() {
  int echantillon = analogRead(MICRO_PIN);
  nivrepos = (nivrepos * (1.0 - ALPHA)) + (echantillon * ALPHA);  // Moyenne exponentielle du niveau moyen
}

//==========================================================
void setup() {
  Serial.begin(115200);
  pinMode(MICRO_PIN, INPUT);
  FastLED.addLeds<WS2812B, LED_PIN, RGB>(led, NB_LEDS);
  FastLED.setMaxPowerInVoltsAndMilliamps(5, 1000);
  FastLED.clear(true);
  FastLED.setBrightness(25);
  delay(500);
  animdepart(); // Animation de démarrage

  // Calibration initiale du niveau de repos
  for (int i = 0; i < 50; i++) {
    majvalrepos();
    delay(10);
  }
}

//==========================================================
void loop() {
  // reactualisation permanente du niveau de repos
  majvalrepos();

  // Lecture de l'amplitude sonore
  int amplitude = mesurepicapic();

  // Lissage de la valeur par calcul de la moyenne exponentielle
  signallisse = signallisse * (1.0 - ALPHA) + amplitude * ALPHA;   

  // Calcul de l'indice des LEDs
  indmic = map(signallisse, 0, 512, 0, NB_LEDS - 1); // Plage ajustée dynamiquement
  indmic = constrain(indmic, 0, NB_LEDS - 1);        // maintien de l'indice dans la bonne plage

  // Mise à jour des LEDs
  for (int i = 0; i < NB_LEDS; i++) {
    if (i < indmic) {
      hue = 30 + (i * 6); // Gradient de couleur
      led[i] = CHSV(hue, 255, 255);
    } 
    else {
      if (i == indmic && indmic > 0) {
          // Gradation pour la dernière LED 
          hue = 30 + (i * 6);
          int brightness = map(signallisse * 100, 0, 512 * 100, 0, 255); // Interpolation
          led[i] = CHSV(hue, 255, brightness);
      } else {
          led[i] = CRGB::Black;
      }
    }
  }
  FastLED.show();

  // Débogage
  #if TRACE
  Serial.print("nivrepos: ");
  Serial.print(nivrepos);
  Serial.print("   Amplitude: ");
  Serial.print(amplitude);
  Serial.print("   Val lissee: ");
  Serial.print(signallisse);
  Serial.print("   indice LEDs: ");
  Serial.println(indmic);
  #endif
  delay(10);  // Petit délai pour stabiliser
}
//_____________________FIN PROG________________________